<?php
/**
 * Interactive Globes Premium
 *
 * @wordpress-plugin
 * Plugin Name: Interactive Globes Premium Pro
 * Plugin URI:        https://wpinteractiveglobes.com/
 * Description:       Create interactive geographic globes. Color full regions or create markers on specific locations that will have information on hover and can also have actions on click. This plugin uses the online globe.gl library to generate the maps.
 * Version:           1.4.2
 * Update URI: https://api.freemius.com
 * Requires PHP:      7.0
 * Author:            Interactive Globes
 * Author URI:        https://wpinteractiveglobes.com/
 * Text Domain:       interactive-globes
 * Domain Path:       /languages
 */

namespace Saltus\WP\Plugin\Saltus\InteractiveGlobes;

use Saltus\WP\Framework\Core as FrameworkCore;
use Saltus\WP\Plugin\Saltus\InteractiveGlobes\Project;
use Saltus\WP\Framework\Infrastructure\Container\ServiceContainer;
use Saltus\WP\Plugin\Saltus\InteractiveGlobes\Plugin\Freemius;
use Saltus\WP\Plugin\Saltus\InteractiveGlobes\Prepare;

/**
 * Step 1: Validate wp and php
 * ######
 */
// If this file is called directly, quit.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Only run plugin code if PHP version bigger than 7.0 for now
if ( version_compare( PHP_VERSION, '7.0', '<' ) ) {
	return;
}

/**
 * Step 2: Check if other plugin variants are active
 * ######
 */
if ( ! function_exists( 'Saltus\WP\Plugin\Saltus\InteractiveGlobes\is_plugin_active' ) ) {
	function is_plugin_active( $plugin ) {
		return in_array( $plugin, (array) get_option( 'active_plugins', array() ), true );
	}
}
$variant_suffix = '-premium/';
if ( strpos( __DIR__, 'premium' ) !== false ) {
	$variant_suffix = '/';
}

$plugin_name         = 'interactive-globes';
$variant_main_dir    = $plugin_name . $variant_suffix;
$variant_plugin_dir  = plugin_dir_path( __DIR__ ) . $variant_main_dir;
$variant_plugin_file = $variant_plugin_dir . 'interactive-globes.php';
$variant_plugin_slug = $plugin_name . $variant_suffix . $plugin_name . '.php';

// run as early as possible
register_activation_hook(
	__FILE__,
	function () use ( $variant_plugin_file, $variant_plugin_slug ) {

		if ( ! file_exists( $variant_plugin_file ) || ! is_plugin_active( $variant_plugin_slug ) ) {
			return;
		}
		// Deactivate the premium plugin.
		\deactivate_plugins( \plugin_basename( $variant_plugin_file ) );
		\add_action(
			'admin_notices',
			function () {
				printf(
					'<div class="notice notice-warning is-dismissible">
						<p>%1$s</p>
					</div>',
					esc_html_e( 'The premium version of the plugin has been deactivated to avoid conflicts', 'interactive-globes' )
				);
			}
		);
	}
);

if ( file_exists( $variant_plugin_file ) && is_plugin_active( $variant_plugin_slug ) ) {
	return;
}

/**
 * Step 3: check if Freemius was already set by other variant
 * ######
 */
global $ig_fs;
// Freemius logic
if ( isset( $ig_fs ) && $ig_fs ) {
	$ig_fs()->set_basename( true, __FILE__ );
	return;
}

/**
 * Step 4: Load config and setup constants
 * ######
 */
if ( ! function_exists( __NAMESPACE__ . '\load_config' ) ) {
	function load_config( $root_dir ) {
		if ( ! file_exists( $root_dir . '/.saltus' ) ) {
			return;
		}
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
		$env   = file_get_contents( $root_dir . '/.saltus' );
		$lines = explode( "\n", $env );

		foreach ( $lines as $line ) {
			preg_match( '/([^#]+)\=(.*)/', $line, $matches );
			if ( isset( $matches[2] ) ) {
				// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.runtime_configuration_putenv
				putenv( trim( $line ) );
			}
		}
	}
}
load_config( __DIR__ );

// before any model
if ( ! defined( 'WP_ENV' ) ) {
	// this is replaced during build time
	define( 'WP_ENV', 'production' );
}

/**
 * Step 5: Load plugin classes
 * ######
 */
if ( file_exists( __DIR__ . '/vendor/autoload.php' ) ) {
	require_once __DIR__ . '/vendor/autoload.php';
}

/**
 * Step 6: Start Freemius and do early Prepare cycle
 * ######
 */
// this is replaced during build time
$fs_is_premium_variant = true;

$fs = null;
// if premium version, setup Freemium
if ( $fs_is_premium_variant ) {
	$fs = new Freemius( __DIR__ );
	$fs->init( $fs_is_premium_variant );

	// make sure if even if its the premium version, the license is valid
	// then do Prepare cycle
	if ( $fs->is_pro() ) {

		$prepare = new Prepare();
		$prepare->register();
	}
}

/**
 * Step 7: Start Plugin
 * ######
 */
if ( ! class_exists( FrameworkCore::class ) ||
	! class_exists( Project::class ) ||
	! class_exists( ServiceContainer::class ) ) {
		return;
}

// loads the modals
$framework = new FrameworkCore( __DIR__ );
$framework->register();

$feature_container  = new ServiceContainer();
$services_container = new ServiceContainer();

$project = new Project( 'interactive-globes', '1.4.2', __FILE__ );

// Initialize plugin
add_action(
	'plugins_loaded',
	function () use ( $project, $framework, $feature_container, $services_container, $fs, $fs_is_premium_variant ) {

		if ( $fs_is_premium_variant && $fs->is_pro() ) {
			$plugin = new CorePro( $project, $framework, $feature_container, $services_container );
		} else {
			$plugin = new Core( $project, $framework, $feature_container, $services_container );
		}

		$plugin->init();
	}
);

/**
 * Step 8: Run Activation cycle
 * ######
 */
// run as early as possible
register_activation_hook( __FILE__, [ __NAMESPACE__ . '\\Plugin\\Lifecycle', 'activation_actions' ] );
